/*
 *  acm : an aerial combat simulator for X - Gallery of common drawings
 *  Copyright (C) 2008  Umberto Salsi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "../V/Vlibmath.h"
#include "../util/memory.h"

#define vpath_gallery_IMPORT
#include "vpath_gallery.h"

static vpath_Type
	*compass_scale = NULL,
	*compass_fixed_scale = NULL,
	*stylized_aircraft = NULL,
	*pointer_arrow = NULL,
	*pointer_triangle = NULL;


vpath_Type * vpath_gallery_get_compass_scale(void)
{
	VMatrix m;
	VPoint a, b;
	int hdg, s_len;
	double fh, fw, margin, r, r_long, r_short, r1;
	char *s;
	static char *labels[] = {"N", "3", "6", "E", "12", "15", "S",
		"21", "24", "W", "30", "33"};
	
	if( compass_scale != NULL )
		return compass_scale;

	compass_scale = vpath_new();

	fh = 0.11;
	fw = fh * 3.0/4.0;
	margin = 0.04;
	r = 1.0;
	r_long = 0.92;
	r_short = 0.96;

	for( hdg = 0; hdg < 360; hdg += 5 ){

		VIdentMatrix(&m);
		VRotate(&m, ZRotation, units_DEGtoRAD(hdg));

		if( hdg % 10 == 0 )
			r1 = r_long;
		else
			r1 = r_short;

		VSetPoint(&a, 0.0, -r, 0.0);
		VSetPoint(&b, 0.0, -r1, 0.0);
		VTransform(&a, &m, &a);
		VTransform(&b, &m, &b);
		vpath_moveTo(compass_scale, &a);
		vpath_lineTo(compass_scale, &b);

		if( hdg % 30 == 0 ){
			
			s = labels[hdg / 30];
			s_len = strlen(s);

			VIdentMatrix(&m);
			VScaleMatrix(&m, fw, fh, 1.0);
			VTranslate(&m, -0.5*fw*s_len, -r1 + margin + fh, 0.0);
			VRotate(&m, ZRotation, units_DEGtoRAD(hdg));
			vpath_draw_string(compass_scale, s, s_len, &m);

		}

	}

	return compass_scale;
}


vpath_Type * vpath_gallery_get_compass_fixed_scale(void)
{
	VMatrix m;
	VPoint a, b;
	int hdg;
	double r, r1;
	
	if( compass_fixed_scale != NULL )
		return compass_fixed_scale;

	compass_fixed_scale = vpath_new();

	r = 1.0;
	r1 = 0.90;

	for( hdg = 0; hdg < 360; hdg += 45 ){

		VIdentMatrix(&m);
		VRotate(&m, ZRotation, units_DEGtoRAD(hdg));

		VSetPoint(&a, 0.0, -r, 0.0);
		VSetPoint(&b, 0.0, -r1, 0.0);
		VTransform(&a, &m, &a);
		VTransform(&b, &m, &b);
		vpath_moveTo(compass_fixed_scale, &a);
		vpath_lineTo(compass_fixed_scale, &b);

	}

	return compass_fixed_scale;
}


vpath_Type * vpath_gallery_get_stylized_aircraft(void)
{
	double l1, l2, l3;
	
	if( stylized_aircraft != NULL )
		return stylized_aircraft;

	stylized_aircraft = vpath_new();

	l1 = 0.14;  /* fuselage half-width */
	l2 = 0.50;  /* elevator half-width */
	l3 = 0.30;  /* wing y offset */

	/* right side: */
	vpath_moveTo(stylized_aircraft, &(VPoint){l1, -1.0, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){l1, 1.0, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){l2, 1.0, 0.0});

	vpath_moveTo(stylized_aircraft, &(VPoint){l1, -l3, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){1.0, -l3, 0.0});

	/* left side (same as above with x reversed): */
	vpath_moveTo(stylized_aircraft, &(VPoint){-l1, -1.0, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){-l1, 1.0, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){-l2, 1.0, 0.0});

	vpath_moveTo(stylized_aircraft, &(VPoint){-l1, -l3, 0.0});
	vpath_lineTo(stylized_aircraft, &(VPoint){-1.0, -l3, 0.0});

	return compass_fixed_scale;
}


vpath_Type * vpath_gallery_get_pointer_arrow(void)
{
	if( pointer_arrow != NULL )
		return pointer_arrow;

	pointer_arrow = vpath_new();

	vpath_moveTo(pointer_arrow, &(VPoint){0.0, -1.0, 0.0});
	vpath_lineTo(pointer_arrow, &(VPoint){0.0, 1.0, 0.0});

	vpath_moveTo(pointer_arrow, &(VPoint){0.0, -1.0, 0.0});
	vpath_lineTo(pointer_arrow, &(VPoint){0.05, -0.80, 0.0});
	vpath_lineTo(pointer_arrow, &(VPoint){0.12, -0.70, 0.0});

	vpath_moveTo(pointer_arrow, &(VPoint){-0.0, -1.0, 0.0});
	vpath_lineTo(pointer_arrow, &(VPoint){-0.05, -0.80, 0.0});
	vpath_lineTo(pointer_arrow, &(VPoint){-0.12, -0.70, 0.0});

	return pointer_arrow;
}


vpath_Type * vpath_gallery_get_pointer_triangle(void)
{
	double l = 0.1;

	if( pointer_triangle != NULL )
		return pointer_triangle;

	pointer_triangle = vpath_new();

	vpath_moveTo(pointer_triangle, &(VPoint){0.0, -1.0, 0.0});
	vpath_lineTo(pointer_triangle, &(VPoint){0.0-l, -1.0-l, 0.0});
	vpath_lineTo(pointer_triangle, &(VPoint){0.0+l, -1.0-l, 0.0});
	vpath_lineTo(pointer_triangle, &(VPoint){0.0, -1.0, 0.0});

	return pointer_triangle;
}


void vpath_gallery_free_all(void)
{
	memory_dispose(compass_scale);
	memory_dispose(compass_fixed_scale);
	memory_dispose(stylized_aircraft);
	memory_dispose(pointer_arrow);
	memory_dispose(pointer_triangle);
}

/* The vpath_gallery.c module end here. */
