/*
 *  $Id: zero_crossing.c 22044 2019-05-23 10:01:55Z yeti-dn $
 *  Copyright (C) 2003-2019 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocess.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define ZERO_CROSSING_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

#define FWHM2SIGMA (1.0/(2.0*sqrt(2*G_LN2)))

typedef enum {
    DISPLAY_DATA,
    DISPLAY_LOG,
    DISPLAY_SHOW
} ZeroCrossingDisplayType;

typedef struct {
    gdouble gaussian_fwhm;
    gdouble threshold;
    gboolean update;
} ZeroCrossingArgs;

typedef struct {
    ZeroCrossingArgs *args;
    GtkWidget *dialog;
    GtkWidget *view;
    GtkObject *gaussian_fwhm;
    GtkObject *threshold;
    GSList *display_group;
    GtkWidget *update;
    GwyContainer *mydata;
    ZeroCrossingDisplayType display;
    gboolean in_init;
    gboolean computed;
    gboolean gauss_computed;
} ZeroCrossingControls;

static gboolean module_register      (void);
static void     zero_crossing        (GwyContainer *data,
                                      GwyRunType run);
static void     zero_crossing_dialog (ZeroCrossingArgs *args,
                                      GwyContainer *data,
                                      GwyDataField *dfield,
                                      gint id,
                                      GQuark squark);
static void     gaussian_fwhm_changed(GtkAdjustment *adj,
                                      ZeroCrossingControls *controls);
static void     threshold_changed    (GtkAdjustment *adj,
                                      ZeroCrossingControls *controls);
static void     display_changed      (GtkToggleButton *radio,
                                      ZeroCrossingControls *controls);
static void     update_changed       (GtkToggleButton *check,
                                      ZeroCrossingControls *controls);
static void     invalidate           (ZeroCrossingControls *controls);
static void     update_controls      (ZeroCrossingControls *controls,
                                      ZeroCrossingArgs *args);
static void     preview              (ZeroCrossingControls *controls,
                                      ZeroCrossingArgs *args);
static void     zero_crossing_run    (const ZeroCrossingArgs *args,
                                      GwyContainer *data,
                                      GwyDataField *dfield,
                                      GQuark squark);
static gdouble  do_log               (GwyDataField *dfield,
                                      GwyDataField *gauss,
                                      gdouble gaussian_fwhm);
static void     do_edge              (GwyDataField *show,
                                      GwyDataField *gauss,
                                      gdouble theshold);
static void     load_args            (GwyContainer *container,
                                      ZeroCrossingArgs *args);
static void     save_args            (GwyContainer *container,
                                      ZeroCrossingArgs *args);

static const ZeroCrossingArgs zero_crossing_defaults = {
    2.0,
    0.1,
    FALSE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Zero crossing edge detection."),
    "Yeti <Yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2019",
};

GWY_MODULE_QUERY2(module_info, zero_crossing)

static gboolean
module_register(void)
{
    gwy_process_func_register("zero_crossing",
                              (GwyProcessFunc)&zero_crossing,
                              N_("/_Presentation/_Edge Detection/_Zero Crossing..."),
                              NULL,
                              ZERO_CROSSING_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Zero crossing step detection presentation"));

    return TRUE;
}

static void
zero_crossing(GwyContainer *data, GwyRunType run)
{
    ZeroCrossingArgs args;
    GwyDataField *dfield;
    GQuark squark;
    gint id;

    g_return_if_fail(run & ZERO_CROSSING_RUN_MODES);
    load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_SHOW_FIELD_KEY, &squark,
                                     0);
    g_return_if_fail(dfield && squark);

    if (run == GWY_RUN_IMMEDIATE)
        zero_crossing_run(&args, data, dfield, squark);
    else {
        zero_crossing_dialog(&args, data, dfield, id, squark);
        save_args(gwy_app_settings_get(), &args);
    }
}

static void
zero_crossing_dialog(ZeroCrossingArgs *args,
                     GwyContainer *data,
                     GwyDataField *dfield,
                     gint id,
                     GQuark squark)
{
    GtkWidget *dialog, *table, *hbox, *label;
    GtkObject *adj;
    ZeroCrossingControls controls;
    gint response;
    GwyDataField *sfield;
    gint row;
    gboolean temp;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.in_init = TRUE;

    dialog = gtk_dialog_new_with_buttons(_("Zero Crossing Step Detection"),
                                         NULL, 0, NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);
    gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.view, FALSE, FALSE, 4);

    table = gtk_table_new(7, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);
    row = 0;

    adj = gtk_adjustment_new(args->gaussian_fwhm, 0.0, 20.0, 0.1, 1.0, 0);
    controls.gaussian_fwhm = adj;
    gwy_table_attach_adjbar(table, row, _("_Gaussian FWHM:"), _("px"),
                            adj, GWY_HSCALE_SQRT);
    g_signal_connect(adj, "value-changed",
                     G_CALLBACK(gaussian_fwhm_changed),
                     &controls);
    row++;

    adj = gtk_adjustment_new(args->threshold, 0.0, 3.0, 0.01, 0.1, 0);
    controls.threshold = adj;
    gwy_table_attach_adjbar(table, row, _("_Threshold:"), _("NRMS"), adj,
                            GWY_HSCALE_SQRT);
    g_signal_connect(adj, "value-changed",
                     G_CALLBACK(threshold_changed), &controls);
    gtk_table_set_row_spacing(GTK_TABLE(table), row, 8);
    row++;

    label = gtk_label_new(_("Display:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.display_group
        = gwy_radio_buttons_createl(G_CALLBACK(display_changed),
                                    &controls, DISPLAY_DATA,
                                    _("Original _image"), DISPLAY_DATA,
                                    _("_LoG convolved"), DISPLAY_LOG,
                                    _("Detected st_ep"), DISPLAY_SHOW,
                                    NULL);
    row = gwy_radio_buttons_attach_to_table(controls.display_group,
                                            GTK_TABLE(table), 2, row);
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);

    controls.update = gtk_check_button_new_with_mnemonic(_("I_nstant updates"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.update),
                                 args->update);
    gtk_table_attach(GTK_TABLE(table), controls.update,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.update, "toggled",
                     G_CALLBACK(update_changed), &controls);

    /* finished initializing, allow instant updates */
    controls.in_init = FALSE;

    /* show initial preview if instant updates are on */
    if (args->update) {
        gtk_dialog_set_response_sensitive(GTK_DIALOG(controls.dialog),
                                          RESPONSE_PREVIEW, FALSE);
    }

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            g_object_unref(controls.mydata);
            return;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            temp = args->update;
            *args = zero_crossing_defaults;
            args->update = temp;
            controls.in_init = TRUE;
            update_controls(&controls, args);
            controls.in_init = FALSE;
            preview(&controls, args);
            break;

            case RESPONSE_PREVIEW:
            preview(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    if (controls.computed) {
        sfield = gwy_container_get_object_by_name(controls.mydata, "/0/show");
        gwy_app_undo_qcheckpointv(data, 1, &squark);
        gwy_container_set_object(data, squark, sfield);
        g_object_unref(controls.mydata);
    }
    else {
        g_object_unref(controls.mydata);
        zero_crossing_run(args, data, dfield, squark);
    }
}

static void
gaussian_fwhm_changed(GtkAdjustment *adj, ZeroCrossingControls *controls)
{
    controls->args->gaussian_fwhm = gtk_adjustment_get_value(adj);
    controls->gauss_computed = FALSE;
    invalidate(controls);
}

static void
threshold_changed(GtkAdjustment *adj, ZeroCrossingControls *controls)
{
    controls->args->threshold = gtk_adjustment_get_value(adj);
    invalidate(controls);
}

static void
display_changed(GtkToggleButton *radio, ZeroCrossingControls *controls)
{
    GwyPixmapLayer *layer;

    if (!gtk_toggle_button_get_active(radio))
        return;

    layer = gwy_data_view_get_base_layer(GWY_DATA_VIEW(controls->view));
    controls->display = gwy_radio_buttons_get_current(controls->display_group);
    preview(controls, controls->args);
    switch (controls->display) {
        case DISPLAY_DATA:
        gwy_pixmap_layer_set_data_key(layer, "/0/data");
        break;

        case DISPLAY_LOG:
        gwy_pixmap_layer_set_data_key(layer, "/0/gauss");
        break;

        case DISPLAY_SHOW:
        gwy_pixmap_layer_set_data_key(layer, "/0/show");
        break;

        default:
        g_return_if_reached();
        break;
    }
}

static void
update_changed(GtkToggleButton *check, ZeroCrossingControls *controls)
{
    controls->args->update = gtk_toggle_button_get_active(check);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      RESPONSE_PREVIEW,
                                      !controls->args->update);
    if (controls->args->update)
        preview(controls, controls->args);
}

static void
invalidate(ZeroCrossingControls *controls)
{
    controls->computed = FALSE;
    if (controls->args->update && !controls->in_init)
        preview(controls, controls->args);
}

static void
update_controls(ZeroCrossingControls *controls, ZeroCrossingArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->gaussian_fwhm),
                             args->gaussian_fwhm);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->threshold),
                             args->threshold);
}

static void
preview(ZeroCrossingControls *controls, ZeroCrossingArgs *args)
{
    GwyDataField *dfield, *show, *gauss;
    gdouble nrms;

    if (controls->computed)
        return;

    dfield = GWY_DATA_FIELD(gwy_container_get_object_by_name(controls->mydata,
                                                             "/0/data"));

    /* Set up the show */
    if (!gwy_container_gis_object_by_name(controls->mydata, "/0/show", &show)) {
        show = create_mask_field(dfield);
        gwy_container_set_object_by_name(controls->mydata, "/0/show", show);
        g_object_unref(show);

        gauss = gwy_data_field_new_alike(show, FALSE);
        gwy_container_set_object_by_name(controls->mydata, "/0/gauss", gauss);
        g_object_unref(gauss);
    }
    else
        gwy_container_gis_object_by_name(controls->mydata, "/0/gauss", &gauss);

    if (controls->gauss_computed)
        nrms = gwy_container_get_double_by_name(controls->mydata, "/0/nrms");
    else {
        nrms = do_log(dfield, gauss, args->gaussian_fwhm);
        gwy_container_set_double_by_name(controls->mydata, "/0/nrms", nrms);
        gwy_data_field_data_changed(gauss);
        controls->gauss_computed = TRUE;
    }
    do_edge(show, gauss, nrms*args->threshold);
    gwy_data_field_data_changed(show);
    controls->computed = TRUE;
}

static void
zero_crossing_run(const ZeroCrossingArgs *args,
                  GwyContainer *data,
                  GwyDataField *dfield,
                  GQuark squark)
{
    GwyDataField *show, *gauss;
    gdouble nrms;

    gwy_app_undo_qcheckpointv(data, 1, &squark);
    show = create_mask_field(dfield);
    gauss = gwy_data_field_new_alike(show, FALSE);
    nrms = do_log(dfield, gauss, args->gaussian_fwhm);
    do_edge(show, gauss, nrms*args->threshold);
    g_object_unref(gauss);
    gwy_container_set_object(data, squark, show);
    g_object_unref(show);
}

static gdouble
do_log(GwyDataField *dfield, GwyDataField *gauss, gdouble gaussian_fwhm)
{
    const gdouble *data, *row;
    gint xres, yres, i, j;
    gdouble nrms;

    gwy_data_field_copy(dfield, gauss, FALSE);
    gwy_data_field_filter_gaussian(gauss, gaussian_fwhm*FWHM2SIGMA);
    gwy_data_field_filter_laplacian(gauss);

    xres = gwy_data_field_get_xres(gauss);
    yres = gwy_data_field_get_yres(gauss);
    data = gwy_data_field_get_data_const(gauss);
    nrms = 0.0;

    for (i = 0; i < yres-1; i++) {
        row = data + i*xres;
        for (j = 0; j < xres; j++)
            nrms += (row[j] - row[j + xres])*(row[j] - row[j + xres]);
    }
    for (i = 0; i < yres; i++) {
        row = data + i*xres;
        for (j = 0; j < xres-1; j++)
            nrms += (row[j] - row[j + 1])*(row[j] - row[j + 1]);
    }

    nrms /= 2*xres*yres - xres - yres;
    return sqrt(nrms);
}

static void
do_edge(GwyDataField *show, GwyDataField *gauss, gdouble threshold)
{
    gdouble *data;
    const gdouble *bdata;
    gdouble dm, dp;
    gint n, xres, yres, i, j;

    gwy_data_field_clear(show);

    xres = gwy_data_field_get_xres(show);
    yres = gwy_data_field_get_yres(show);
    data = gwy_data_field_get_data(show);
    bdata = gwy_data_field_get_data_const(gauss);

    /* Vertical pass */
    for (i = 1; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            n = i*xres + j;
            dm = bdata[n - xres];
            dp = bdata[n];
            if (dm*dp <= 0.0) {
                dm = fabs(dm);
                dp = fabs(dp);
                if (dm >= threshold || dp >= threshold) {
                    if (dm < dp)
                        data[n - xres] = 1.0;
                    else if (dp < dm)
                        data[n] = 1.0;
                    /* If they are equal and different from zero, sigh and
                     * choose an arbitrary one */
                    else if (dm > 0.0)
                        data[n] = 1.0;
                }
            }
        }
    }

    /* Horizontal pass */
    for (i = 0; i < yres; i++) {
        for (j = 1; j < xres; j++) {
            n = i*xres + j;
            dm = bdata[n - 1];
            dp = bdata[n];
            if (dm*dp <= 0.0) {
                dm = fabs(dm);
                dp = fabs(dp);
                if (dm >= threshold || dp >= threshold) {
                    if (dm < dp)
                        data[n - 1] = 1.0;
                    else if (dp < dm)
                        data[n] = 1.0;
                    /* If they are equal and different from zero, sigh and
                     * choose an arbitrary one */
                    else if (dm > 0.0)
                        data[n] = 1.0;
                }
            }
        }
    }
}

static const gchar gaussian_fwhm_key[] = "/module/zero_crossing/gaussian-fwhm";
static const gchar threshold_key[]     = "/module/zero_crossing/threshold";
static const gchar update_key[]        = "/module/zero_crossing/update";

static void
sanitize_args(ZeroCrossingArgs *args)
{
    args->gaussian_fwhm = CLAMP(args->gaussian_fwhm, 0.0, 20.0);
    args->threshold = CLAMP(args->threshold, 0.0, 3.0);
    args->update = !!args->update;
}

static void
load_args(GwyContainer *container, ZeroCrossingArgs *args)
{
    *args = zero_crossing_defaults;

    gwy_container_gis_double_by_name(container, gaussian_fwhm_key,
                                     &args->gaussian_fwhm);
    gwy_container_gis_double_by_name(container, threshold_key,
                                     &args->threshold);
    gwy_container_gis_boolean_by_name(container, update_key,
                                      &args->update);

    sanitize_args(args);
}

static void
save_args(GwyContainer *container, ZeroCrossingArgs *args)
{
    gwy_container_set_double_by_name(container, gaussian_fwhm_key,
                                     args->gaussian_fwhm);
    gwy_container_set_double_by_name(container, threshold_key,
                                     args->threshold);
    gwy_container_set_boolean_by_name(container, update_key,
                                      args->update);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
