#!/usr/bin/env python

"""
A simple filesystem-based store of calendar data.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.stores.common import StoreBase, PublisherBase, JournalBase

from datetime import datetime
from imiptools.config import settings
from imiptools.data import make_calendar, parse_object, to_stream
from imiptools.dates import format_datetime, get_datetime, to_timezone
from imiptools.filesys import fix_permissions, FileBase
from imiptools.period import FreeBusyPeriod, FreeBusyGroupPeriod, \
                             FreeBusyOfferPeriod, FreeBusyCollection, \
                             FreeBusyGroupCollection, FreeBusyOffersCollection
from imiptools.text import get_table, set_defaults
from os.path import isdir, isfile, join
from os import listdir, remove, rmdir
import codecs

STORE_DIR = settings["STORE_DIR"]
PUBLISH_DIR = settings["PUBLISH_DIR"]
JOURNAL_DIR = settings["JOURNAL_DIR"]

class FileStoreBase(FileBase):

    "A file store supporting user-specific locking and tabular data."

    def acquire_lock(self, user, timeout=None):
        FileBase.acquire_lock(self, timeout, user)

    def release_lock(self, user):
        FileBase.release_lock(self, user)

    # Utility methods.

    def _set_defaults(self, t, empty_defaults):
        return set_defaults(t, empty_defaults)

    def _get_table(self, filename, empty_defaults=None, tab_separated=True):

        """
        From the file having the given 'filename', return a list of tuples
        representing the file's contents.

        The 'empty_defaults' is a list of (index, value) tuples indicating the
        default value where a column either does not exist or provides an empty
        value.

        If 'tab_separated' is specified and is a false value, line parsing using
        the imiptools.text.parse_line function will be performed instead of
        splitting each line of the file using tab characters as separators.
        """

        return get_table(filename, empty_defaults, tab_separated)

    def _get_table_atomic(self, user, filename, empty_defaults=None, tab_separated=True):

        """
        From the file for the given 'user' having the given 'filename', return
        a list of tuples representing the file's contents.

        The 'empty_defaults' is a list of (index, value) tuples indicating the
        default value where a column either does not exist or provides an empty
        value.

        If 'tab_separated' is specified and is a false value, line parsing using
        the imiptools.text.parse_line function will be performed instead of
        splitting each line of the file using tab characters as separators.
        """

        self.acquire_lock(user)
        try:
            return self._get_table(filename, empty_defaults, tab_separated)
        finally:
            self.release_lock(user)

    def _set_table(self, filename, items, empty_defaults=None):

        """
        Write to the file having the given 'filename' the 'items'.

        The 'empty_defaults' is a list of (index, value) tuples indicating the
        default value where a column either does not exist or provides an empty
        value.
        """

        f = codecs.open(filename, "wb", encoding="utf-8")
        try:
            for item in items:
                self._set_table_item(f, item, empty_defaults)
        finally:
            f.close()
            fix_permissions(filename)

    def _set_table_item(self, f, item, empty_defaults=None):

        "Set in table 'f' the given 'item', using any 'empty_defaults'."

        if empty_defaults:
            item = self._set_defaults(list(item), empty_defaults)
        f.write("\t".join(item) + "\n")

    def _set_table_atomic(self, user, filename, items, empty_defaults=None):

        """
        For the given 'user', write to the file having the given 'filename' the
        'items'.

        The 'empty_defaults' is a list of (index, value) tuples indicating the
        default value where a column either does not exist or provides an empty
        value.
        """

        self.acquire_lock(user)
        try:
            self._set_table(filename, items, empty_defaults)
        finally:
            self.release_lock(user)

    def _set_freebusy(self, user, freebusy, filename):

        """
        For the given 'user', convert the 'freebusy' details to a form suitable
        for writing to 'filename'.
        """

        # Obtain tuples from the free/busy objects.

        self._set_table_atomic(user, filename,
            map(lambda fb: freebusy.make_tuple(fb.as_tuple(strings_only=True)), list(freebusy)))

class Store(FileStoreBase, StoreBase):

    "A file store of tabular free/busy data and objects."

    def __init__(self, store_dir=None):
        FileBase.__init__(self, store_dir or STORE_DIR)

    # Store object access.

    def _get_object(self, user, filename):

        """
        Return the parsed object for the given 'user' having the given
        'filename'.
        """

        self.acquire_lock(user)
        try:
            f = open(filename, "rb")
            try:
                return parse_object(f, "utf-8")
            finally:
                f.close()
        finally:
            self.release_lock(user)

    def _set_object(self, user, filename, node):

        """
        Set an object for the given 'user' having the given 'filename', using
        'node' to define the object.
        """

        self.acquire_lock(user)
        try:
            f = open(filename, "wb")
            try:
                to_stream(f, node)
            finally:
                f.close()
                fix_permissions(filename)
        finally:
            self.release_lock(user)

        return True

    def _remove_object(self, filename):

        "Remove the object with the given 'filename'."

        try:
            remove(filename)
        except OSError:
            return False

        return True

    def _remove_collection(self, filename):

        "Remove the collection with the given 'filename'."

        try:
            rmdir(filename)
        except OSError:
            return False

        return True

    # User discovery.

    def get_users(self):

        "Return a list of users."

        return listdir(self.store_dir)

    # Event and event metadata access.

    def get_events(self, user):

        "Return a list of event identifiers."

        filename = self.get_object_in_store(user, "objects")
        if not filename or not isdir(filename):
            return []

        return [name for name in listdir(filename) if isfile(join(filename, name))]

    def get_cancelled_events(self, user):

        "Return a list of event identifiers for cancelled events."

        filename = self.get_object_in_store(user, "cancellations", "objects")
        if not filename or not isdir(filename):
            return []

        return [name for name in listdir(filename) if isfile(join(filename, name))]

    def get_event(self, user, uid, recurrenceid=None, dirname=None):

        """
        Get the event for the given 'user' with the given 'uid'. If
        the optional 'recurrenceid' is specified, a specific instance or
        occurrence of an event is returned.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, dirname)
        if not filename or not isfile(filename):
            return None

        return filename and self._get_object(user, filename)

    def get_complete_event(self, user, uid):

        "Get the event for the given 'user' with the given 'uid'."

        filename = self.get_complete_event_filename(user, uid)
        if not filename or not isfile(filename):
            return None

        return filename and self._get_object(user, filename)

    def set_complete_event(self, user, uid, node):

        "Set an event for 'user' having the given 'uid' and 'node'."

        filename = self.get_object_in_store(user, "objects", uid)
        if not filename:
            return False

        return self._set_object(user, filename, node)

    def remove_parent_event(self, user, uid):

        "Remove the parent event for 'user' having the given 'uid'."

        filename = self.get_object_in_store(user, "objects", uid)
        if not filename:
            return False

        return self._remove_object(filename)

    def get_recurrences(self, user, uid):

        """
        Get additional event instances for an event of the given 'user' with the
        indicated 'uid'. Both active and cancelled recurrences are returned.
        """

        return self.get_active_recurrences(user, uid) + self.get_cancelled_recurrences(user, uid)

    def get_active_recurrences(self, user, uid):

        """
        Get additional event instances for an event of the given 'user' with the
        indicated 'uid'. Cancelled recurrences are not returned.
        """

        filename = self.get_object_in_store(user, "recurrences", uid)
        if not filename or not isdir(filename):
            return []

        return [name for name in listdir(filename) if isfile(join(filename, name))]

    def get_cancelled_recurrences(self, user, uid):

        """
        Get additional event instances for an event of the given 'user' with the
        indicated 'uid'. Only cancelled recurrences are returned.
        """

        filename = self.get_object_in_store(user, "cancellations", "recurrences", uid)
        if not filename or not isdir(filename):
            return []

        return [name for name in listdir(filename) if isfile(join(filename, name))]

    def get_recurrence(self, user, uid, recurrenceid):

        """
        For the event of the given 'user' with the given 'uid', return the
        specific recurrence indicated by the 'recurrenceid'.
        """

        filename = self.get_recurrence_filename(user, uid, recurrenceid)
        if not filename or not isfile(filename):
            return None

        return filename and self._get_object(user, filename)

    def set_recurrence(self, user, uid, recurrenceid, node):

        "Set an event for 'user' having the given 'uid' and 'node'."

        filename = self.get_object_in_store(user, "recurrences", uid, recurrenceid)
        if not filename:
            return False

        return self._set_object(user, filename, node)

    def remove_recurrence(self, user, uid, recurrenceid):

        """
        Remove a special recurrence from an event stored by 'user' having the
        given 'uid' and 'recurrenceid'.
        """

        filename = self.get_object_in_store(user, "recurrences", uid, recurrenceid)
        if not filename:
            return False

        return self._remove_object(filename)

    def remove_recurrence_collection(self, user, uid):

        """
        Remove the collection of recurrences stored by 'user' having the given
        'uid'.
        """

        recurrences = self.get_object_in_store(user, "recurrences", uid)
        if recurrences:
            return self._remove_collection(recurrences)

        return True

    # Event filename computation.

    def get_event_filename(self, user, uid, recurrenceid=None, dirname=None, username=None):

        """
        Get the filename providing the event for the given 'user' with the given
        'uid'. If the optional 'recurrenceid' is specified, a specific instance
        or occurrence of an event is returned.

        Where 'dirname' is specified, the given directory name is used as the
        base of the location within which any filename will reside.
        """

        if recurrenceid:
            return self.get_recurrence_filename(user, uid, recurrenceid, dirname, username)
        else:
            return self.get_complete_event_filename(user, uid, dirname, username)

    def get_recurrence_filename(self, user, uid, recurrenceid, dirname=None, username=None):

        """
        For the event of the given 'user' with the given 'uid', return the
        filename providing the recurrence with the given 'recurrenceid'.

        Where 'dirname' is specified, the given directory name is used as the
        base of the location within which any filename will reside.

        Where 'username' is specified, the event details will reside in a file
        bearing that name within a directory having 'uid' as its name.
        """

        return self.get_object_in_store(user, dirname, "recurrences", uid, recurrenceid, username)

    def get_complete_event_filename(self, user, uid, dirname=None, username=None):

        """
        Get the filename providing the event for the given 'user' with the given
        'uid'. 

        Where 'dirname' is specified, the given directory name is used as the
        base of the location within which any filename will reside.

        Where 'username' is specified, the event details will reside in a file
        bearing that name within a directory having 'uid' as its name.
        """

        return self.get_object_in_store(user, dirname, "objects", uid, username)

    # Free/busy period providers, upon extension of the free/busy records.

    def _get_freebusy_providers(self, user):

        """
        Return the free/busy providers for the given 'user'.

        This function returns any stored datetime and a list of providers as a
        2-tuple. Each provider is itself a (uid, recurrenceid) tuple.
        """

        filename = self.get_object_in_store(user, "freebusy-providers")
        if not filename or not isfile(filename):
            return None

        # Attempt to read providers, with a declaration of the datetime
        # from which such providers are considered as still being active.

        t = self._get_table_atomic(user, filename, [(1, None)])
        try:
            dt_string = t[0][0]
        except IndexError:
            return None

        return dt_string, t[1:]

    def _set_freebusy_providers(self, user, dt_string, t):

        "Set the given provider timestamp 'dt_string' and table 't'."

        filename = self.get_object_in_store(user, "freebusy-providers")
        if not filename:
            return False

        t.insert(0, (dt_string,))
        self._set_table_atomic(user, filename, t, [(1, "")])
        return True

    # Free/busy period access.

    def get_freebusy(self, user, name=None, mutable=False, cls=None):

        "Get free/busy details for the given 'user'."

        filename = self.get_object_in_store(user, name or "freebusy")

        if not filename or not isfile(filename):
            periods = []
        else:
            cls = cls or FreeBusyPeriod
            periods = map(lambda t: cls(*t),
                self._get_table_atomic(user, filename))

        return FreeBusyCollection(periods, mutable)

    def get_freebusy_for_other(self, user, other, mutable=False, cls=None, collection=None):

        "For the given 'user', get free/busy details for the 'other' user."

        filename = self.get_object_in_store(user, "freebusy-other", other)

        if not filename or not isfile(filename):
            periods = []
        else:
            cls = cls or FreeBusyPeriod
            periods = map(lambda t: cls(*t),
                self._get_table_atomic(user, filename))

        collection = collection or FreeBusyCollection
        return collection(periods, mutable)

    def set_freebusy(self, user, freebusy, name=None):

        "For the given 'user', set 'freebusy' details."

        filename = self.get_object_in_store(user, name or "freebusy")
        if not filename:
            return False

        self._set_freebusy(user, freebusy, filename)
        return True

    def set_freebusy_for_other(self, user, freebusy, other):

        "For the given 'user', set 'freebusy' details for the 'other' user."

        filename = self.get_object_in_store(user, "freebusy-other", other)
        if not filename:
            return False

        self._set_freebusy(user, freebusy, filename)
        return True

    def get_freebusy_others(self, user):

        """
        For the given 'user', return a list of other users for whom free/busy
        information is retained.
        """

        filename = self.get_object_in_store(user, "freebusy-other")

        if not filename or not isdir(filename):
            return []

        return listdir(filename)

    # Tentative free/busy periods related to countering.

    def get_freebusy_offers(self, user, mutable=False):

        "Get free/busy offers for the given 'user'."

        offers = []
        expired = []
        now = to_timezone(datetime.utcnow(), "UTC")

        # Expire old offers and save the collection if modified.

        self.acquire_lock(user)
        try:
            l = self.get_freebusy(user, "freebusy-offers", cls=FreeBusyOfferPeriod)
            for fb in l:
                if fb.expires and get_datetime(fb.expires) <= now:
                    expired.append(fb)
                else:
                    offers.append(fb)

            if expired:
                self.set_freebusy_offers(user, offers)
        finally:
            self.release_lock(user)

        return FreeBusyOffersCollection(offers, mutable)

    # Requests and counter-proposals.

    def _get_requests(self, user, queue):

        "Get requests for the given 'user' from the given 'queue'."

        filename = self.get_object_in_store(user, queue)
        if not filename or not isfile(filename):
            return []

        return self._get_table_atomic(user, filename, [(1, None), (2, None)])

    def get_requests(self, user):

        "Get requests for the given 'user'."

        return self._get_requests(user, "requests")

    def _set_requests(self, user, requests, queue):

        """
        For the given 'user', set the list of queued 'requests' in the given
        'queue'.
        """

        filename = self.get_object_in_store(user, queue)
        if not filename:
            return False

        self._set_table_atomic(user, filename, requests, [(1, ""), (2, "")])
        return True

    def set_requests(self, user, requests):

        "For the given 'user', set the list of queued 'requests'."

        return self._set_requests(user, requests, "requests")

    def _set_request(self, user, request, queue):

        """
        For the given 'user', set the given 'request' in the given 'queue'.
        """

        filename = self.get_object_in_store(user, queue)
        if not filename:
            return False

        self.acquire_lock(user)
        try:
            f = codecs.open(filename, "ab", encoding="utf-8")
            try:
                self._set_table_item(f, request, [(1, ""), (2, "")])
            finally:
                f.close()
                fix_permissions(filename)
        finally:
            self.release_lock(user)

        return True

    def set_request(self, user, uid, recurrenceid=None, type=None):

        """
        For the given 'user', set the queued 'uid' and 'recurrenceid',
        indicating a request, along with any given 'type'.
        """

        return self._set_request(user, (uid, recurrenceid, type), "requests")

    def get_counters(self, user, uid, recurrenceid=None):

        """
        For the given 'user', return a list of users from whom counter-proposals
        have been received for the given 'uid' and optional 'recurrenceid'.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, "counters")
        if not filename or not isdir(filename):
            return []

        return [name for name in listdir(filename) if isfile(join(filename, name))]

    def get_counter(self, user, other, uid, recurrenceid=None):

        """
        For the given 'user', return the counter-proposal from 'other' for the
        given 'uid' and optional 'recurrenceid'.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, "counters", other)
        if not filename or not isfile(filename):
            return None

        return self._get_object(user, filename)

    def set_counter(self, user, other, node, uid, recurrenceid=None):

        """
        For the given 'user', store a counter-proposal received from 'other' the
        given 'node' representing that proposal for the given 'uid' and
        'recurrenceid'.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, "counters", other)
        if not filename:
            return False

        return self._set_object(user, filename, node)

    def remove_counters(self, user, uid, recurrenceid=None):

        """
        For the given 'user', remove all counter-proposals associated with the
        given 'uid' and 'recurrenceid'.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, "counters")
        if not filename or not isdir(filename):
            return False

        removed = False

        for other in listdir(filename):
            counter_filename = self.get_event_filename(user, uid, recurrenceid, "counters", other)
            removed = removed or self._remove_object(counter_filename)

        return removed

    def remove_counter(self, user, other, uid, recurrenceid=None):

        """
        For the given 'user', remove any counter-proposal from 'other'
        associated with the given 'uid' and 'recurrenceid'.
        """

        filename = self.get_event_filename(user, uid, recurrenceid, "counters", other)
        if not filename or not isfile(filename):
            return False

        return self._remove_object(filename)

    # Event cancellation.

    def cancel_event(self, user, uid, recurrenceid=None):

        """
        Cancel an event for 'user' having the given 'uid'. If the optional
        'recurrenceid' is specified, a specific instance or occurrence of an
        event is cancelled.
        """

        filename = self.get_event_filename(user, uid, recurrenceid)
        cancelled_filename = self.get_event_filename(user, uid, recurrenceid, "cancellations")

        if filename and cancelled_filename and isfile(filename):
            return self.move_object(filename, cancelled_filename)

        return False

    def uncancel_event(self, user, uid, recurrenceid=None):

        """
        Uncancel an event for 'user' having the given 'uid'. If the optional
        'recurrenceid' is specified, a specific instance or occurrence of an
        event is uncancelled.
        """

        filename = self.get_event_filename(user, uid, recurrenceid)
        cancelled_filename = self.get_event_filename(user, uid, recurrenceid, "cancellations")

        if filename and cancelled_filename and isfile(cancelled_filename):
            return self.move_object(cancelled_filename, filename)

        return False

    def remove_cancellation(self, user, uid, recurrenceid=None):

        """
        Remove a cancellation for 'user' for the event having the given 'uid'.
        If the optional 'recurrenceid' is specified, a specific instance or
        occurrence of an event is affected.
        """

        # Remove any parent event cancellation or a specific recurrence
        # cancellation if indicated.

        filename = self.get_event_filename(user, uid, recurrenceid, "cancellations")

        if filename and isfile(filename):
            return self._remove_object(filename)

        return False

class Publisher(FileBase, PublisherBase):

    "A publisher of objects."

    def __init__(self, store_dir=None):
        FileBase.__init__(self, store_dir or PUBLISH_DIR)

    def set_freebusy(self, user, freebusy):

        "For the given 'user', set 'freebusy' details."

        filename = self.get_object_in_store(user, "freebusy")
        if not filename:
            return False

        record = []
        rwrite = record.append

        rwrite(("ORGANIZER", {}, user))
        rwrite(("UID", {}, user))
        rwrite(("DTSTAMP", {}, datetime.utcnow().strftime("%Y%m%dT%H%M%SZ")))

        for fb in freebusy:
            if not fb.transp or fb.transp == "OPAQUE":
                rwrite(("FREEBUSY", {"FBTYPE" : "BUSY"}, "/".join(
                    map(format_datetime, [fb.get_start_point(), fb.get_end_point()]))))

        f = open(filename, "wb")
        try:
            to_stream(f, make_calendar([("VFREEBUSY", {}, record)], "PUBLISH"))
        finally:
            f.close()
            fix_permissions(filename)

        return True

class Journal(Store, JournalBase):

    "A journal system to support quotas."

    # Quota and user identity/group discovery.

    get_quotas = Store.get_users
    get_quota_users = Store.get_freebusy_others

    # Delegate information for the quota.

    def get_delegates(self, quota):

        "Return a list of delegates for 'quota'."

        filename = self.get_object_in_store(quota, "delegates")
        if not filename or not isfile(filename):
            return []

        return [value for (value,) in self._get_table_atomic(quota, filename)]

    def set_delegates(self, quota, delegates):

        "For the given 'quota', set the list of 'delegates'."

        filename = self.get_object_in_store(quota, "delegates")
        if not filename:
            return False

        self._set_table_atomic(quota, filename, [(value,) for value in delegates])
        return True

    # Groups of users sharing quotas.

    def get_groups(self, quota):

        "Return the identity mappings for the given 'quota' as a dictionary."

        filename = self.get_object_in_store(quota, "groups")
        if not filename or not isfile(filename):
            return {}

        return dict(self._get_table_atomic(quota, filename, tab_separated=False))

    def set_groups(self, quota, groups):

        "For the given 'quota', set 'groups' mapping users to groups."

        filename = self.get_object_in_store(quota, "groups")
        if not filename:
            return False

        self._set_table_atomic(quota, filename, groups.items())
        return True

    def get_limits(self, quota):

        """
        Return the limits for the 'quota' as a dictionary mapping identities or
        groups to durations.
        """

        filename = self.get_object_in_store(quota, "limits")
        if not filename or not isfile(filename):
            return {}

        return dict(self._get_table_atomic(quota, filename, tab_separated=False))

    def set_limits(self, quota, limits):

        """
        For the given 'quota', set the given 'limits' on resource usage mapping
        groups to limits.
        """

        filename = self.get_object_in_store(quota, "limits")
        if not filename:
            return False

        self._set_table_atomic(quota, filename, limits.items())
        return True

    # Journal entry methods.

    def get_entries(self, quota, group, mutable=False):

        """
        Return a list of journal entries for the given 'quota' for the indicated
        'group'.
        """

        return self.get_freebusy_for_other(quota, group, mutable)

    def set_entries(self, quota, group, entries):

        """
        For the given 'quota' and indicated 'group', set the list of journal
        'entries'.
        """

        return self.set_freebusy_for_other(quota, entries, group)

    # Compatibility methods.

    def get_freebusy_for_other(self, user, other, mutable=False):
        return Store.get_freebusy_for_other(self, user, other, mutable, cls=FreeBusyGroupPeriod, collection=FreeBusyGroupCollection)

# vim: tabstop=4 expandtab shiftwidth=4
