if (typeof T === 'undefined') require('../setup');

T('integer pow', function () {

  // Highest finite exponent here is 100
  function t(expected, n, exp) {
    T.assertEqual(expected, new Decimal(n).pow(exp).valueOf());
  }

  function tx(fn, msg) {
    T.assertException(fn, msg);
  }

  Decimal.config({
    precision: 40,
    rounding: 4,
    toExpNeg: -7,
    toExpPos: 21,
    minE: -9e15,
    maxE: 9e15
  });

  // Test the special cases specified by ES 15.8.2.13 (from v8)

  t('4', 2, 2);
  t('2147483648', 2, 31);
  t('0.25', 2, -2);
  t('0.0625', 2, -4);
  t('1', 1, 100);
  t('0', 0, 1000);

  t('NaN', 2, NaN);
  t('NaN', +0, NaN);
  t('NaN', -0, NaN);
  t('NaN', Infinity, NaN);
  t('NaN', -Infinity, NaN);

  t('1', NaN, +0);
  t('1', NaN, -0);

  t('NaN', NaN, NaN);
  t('NaN', NaN, 2.2);
  t('NaN', NaN, 1);
  t('NaN', NaN, -1);
  t('NaN', NaN, -2.2);
  t('NaN', NaN, Infinity);
  t('NaN', NaN, -Infinity);

  t('Infinity', 1.1, Infinity);
  t('Infinity', -1.1, Infinity);
  t('Infinity', 2, Infinity);
  t('Infinity', -2, Infinity);
  t('0', 2, -Infinity);
  t('0', -2, -Infinity);

  t('Infinity', 1/1.1, -Infinity);
  t('Infinity', 1/-1.1, -Infinity);
  t('Infinity', 1/2, -Infinity);
  t('Infinity', 1/-2, -Infinity);

  t('NaN', 1, Infinity);
  t('NaN', 1, -Infinity);
  t('NaN', -1, Infinity);
  t('NaN', -1, -Infinity);

  t('0', 0.1, Infinity);
  t('0', -0.1, Infinity);
  t('0', 0.999, Infinity);
  t('0', -0.999, Infinity);

  t('Infinity', 0.1, -Infinity);
  t('Infinity', -0.1, -Infinity);
  t('Infinity', 0.999, -Infinity);
  t('Infinity', -0.999, -Infinity);

  t('Infinity', Infinity, 2);
  t('Infinity', 1/Infinity, -2);
  t('Infinity', -Infinity, 2);

  t('-Infinity', -Infinity, 3);
  t('-Infinity', -Infinity, 13);

  t('-Infinity', 1/-Infinity, -3);
  t('-Infinity', 1/-Infinity, -13);
  t('-Infinity', 1/-0, 3);
  t('-Infinity', 1/-0, 13);
  t('-Infinity', -0, -3);
  t('-Infinity', -0, -13);
  t('-Infinity', -0, -1);

  t('-Infinity', -0, -10000000001);
  t('Infinity', 1/-Infinity, -2);

  t('Infinity', 1/+0, 1.1);
  t('Infinity', 1/+0, 2);

  t('Infinity', +0, -1.1);
  t('Infinity', +0, -2);

  t('Infinity', 1/-0, 2);
  t('Infinity', -0, -2);

  t('NaN', -0.00001, 1.1);
  t('NaN', -0.00001, -1.1);

  t('-Infinity', 1/-0, 1);
  t('-Infinity', 1/-0, 10000000001);

  t('NaN', 2, NaN);

  t('Infinity', 1/-0, 0.5);
  t('Infinity', -0, -0.5);
  t('Infinity', 1/-0, 0.6);
  t('Infinity', -0, -0.6);
  t('Infinity', Infinity, 0.1);
  t('Infinity', 1/Infinity, -0.1);

  Decimal.precision = 20;

  t('1.9801312458591796501e+301030', 2, 1e6 + 1);      // Math.pow( 2, 1e6 + 1) == Infinity
  t('5.0501702959901511235e-301031', 2, -1e6 - 1);     // Math.pow( 2, -1e6 - 1) == 0
  t('-1.9801312458591796501e+301030', -2, 1e6 + 1);    // Math.pow(-2, 1e6 + 1) == -Infinity
  t('-5.0501702959901511235e-301031', -2, -1e6 - 1);   // Math.pow(-2, -1e6 - 1) == -0
  t('-1.9801312458591796501e-698971', -0.2, 1e6 + 1);  // Math.pow(-0.2, 1e6 + 1) == -0
  t('-5.0501702959901511235e+698970', -0.2, -1e6 - 1); // Math.pow(-0.2, -1e6 - 1) == -Infinity
  t('1.9801312458591796501e-698971', 0.2, 1e6 + 1);    // Math.pow( 0.2, 1e6 + 1) == 0
  t('5.0501702959901511235e+698970', 0.2, -1e6 - 1);   // Math.pow( 0.2, -1e6 - 1) == Infinity

  Decimal.precision = 600;

  t('4096', '8', 4);
  t('-1.331', '-1.1', 3);
  t('5.125696', '-2.264', 2);
  t('6.7266e-7', '6.7266E-7', 1);
  t('4142779.6499215776', '21.06', 5);
  t('731.1616', '-5.2', 4);
  t('1', '61818', 0);
  t('3.2', '3.2', 1);
  t('5.4139923025768140625e-41', '-1.945E-7', 6);
  t('1280630.81718016', '5.8', 8);
  t('3965.318943552', '15.828', 3);
  t('53.721', '53.721', 1);
  t('1', '-1.9', 0);
  t('4.58357323731267363492522744606954913014016e-39', '-0.0000161306', 8);
  t('-8560814308.6108448224', '-96.94', 5);
  t('4.4127502627834341562081e-74', '-6.77E-10', 8);
  t('79327261142.56790234534719652175488087744161', '23.0371', 8);
  t('3.101121e-38', '1.761E-19', 2);
  t('1', '-1', 4);
  t('6.23201296e-28', '-1.58E-7', 4);
  t('-8.50893402176e-19', '-9.476E-7', 3);
  t('16', '4', 2);
  t('90368789.0625', '-97.5', 4);
  t('1', '-112.8', 0);
  t('4.122181458338334221291398681640625e+27', '40042.5', 6);
  t('5.94467302491009e+21', '1290.0', 7);
  t('5.6599836943004175019970957e+25', '141437', 5);
  t('69.75757441', '1.7', 8);
  t('1e-42', '0.0000010', 7);

  t('8', '8', 1);
  t('-2.5090696333749305038864384e+25', '-664', 9);
  t('24794.911296', '5.4', 6);
  t('7077398515.1515538432', '93.32', 5);
  t('-1.4520042511984659693722813984375e-130', '-2.8295E-19', 7);
  t('4', '4', 1);
  t('372088627687.312953646321', '781.019', 4);
  t('-5.3864523289963490660381317787272961329536e-23', '-0.000658806', 7);
  t('-1.8', '-1.8', 1);
  t('1', '-5.9', 0);
  t('1', '2', 0);
  t('-6.4097340625', '-1.45', 5);
  t('170859375', '15', 7);
  t('595732589817199.440265999920799232', '43.82', 9);
  t('1.7080198121677824e+36', '-4200', 10);
  t('1', '658.8', 0);
  t('136.460505366756569881', '2.269', 6);
  t('-1', '-1', 3);
  t('236007.159691873761', '-22.041', 4);
  t('1', '-1.1', 0);
  t('287803125756.085809269657829376', '18.76', 9);
  t('4.1069049025e-34', '-2.02655E-17', 2);
  t('-8', '-2', 3);
  t('1704883919.2576', '203.2', 4);
  t('22106814.0740608', '11.2', 7);
  t('3.7481851264119295287828498195966142655968201e+23', '227.71', 10);
  t('-3', '-3', 1);
  t('5.308416e-54', '4.8E-14', 4);
  t('1', '-1', 8);
  t('1', '-4.1', 0);
  t('398', '398', 1);
  t('1', '1.17981E-18', 0);
  t('255896509713547.45824', '761.4', 5);
  t('16807', '7', 5);
  t('1', '232.6', 0);
  t('2.67066142562472466573674890357652039841e+30', '-6358.1', 8);
  t('390625', '-5', 8);
  t('85766121', '21.0', 6);
  t('2206809.987903620081317314341735861401', '4.309', 10);
  t('6.1917364224e-80', '-1.2E-8', 10);
  t('64', '-2', 6);
  t('-39.8778220049', '-2.09', 5);
  t('892496818.033068251283537321', '-31.029', 6);
  t('1.1289646949223432899980166202016362758071452681629968849e+25', '320.087', 10);
  t('1932991.455312009', '124.569', 3);
  t('635307227133823.258624', '-293.2', 6);
  t('784', '-28', 2);
  t('-43361725294765860565.175323475675341312', '-152.02', 9);
  t('1589.6036888689492933115234375', '4.36775', 5);
  t('1', '-18.4', 0);
  t('20832.8532313216810321678408500840248440324096', '3.46612', 8);
  t('6.5536', '-1.6', 4);
  t('174729381067.247616', '5590.56', 3);
  t('-4.084101e-14', '-0.0021', 5);
  t('7.46848810699576790739263937619996819197249e+21', '153.93', 10);
  t('2.09881105970752e-16', '0.0000059428', 3);
  t('23298085122481', '-169', 6);
  t('1', '-4.095E-11', 0);
  t('30016915.773120638290557721', '17.629', 6);
  t('16807', '7', 5);
  t('130666515.5580240243056896', '-10.34', 8);
  t('-5.97080224872032e-91', '-9.02E-19', 5);
  t('-3450.25251', '-5.1', 5);
  t('43046721', '-9', 8);
  t('1', '1', 1);
  t('3545.66529682492339392399', '5.1279', 5);
  t('3396.171616714297', '15.0313', 3);
  t('4.622674003397519975764019459730496e+33', '-408146', 6);
  t('3.24210716131761936e-27', '2.38620E-7', 4);
  t('9', '3', 2);
  t('14641', '121', 2);
  t('62.81386652381601821841', '2.81523', 4);
  t('1', '8', 0);
  t('4.29981696e-160', '-1.2E-20', 8);
  t('1.692652673311521044295612689449216e-7', '-0.14242', 8);
  t('1', '-19.5', 0);
  t('1', '8621.8', 0);
  t('7.907e-7', '7.907E-7', 1);
  t('1.61051', '1.1', 5);
  t('1.553e-14', '1.553E-14', 1);
  t('858461358961485360081', '171171', 4);
  t('64', '-2', 6);
  t('-2.9', '-2.9', 1);
  t('-2.0661046784e-179', '-1.4E-20', 9);
  t('39.0625', '-2.5', 4);
  t('-391460020121.8781037', '-45.3', 7);
  t('-80.1', '-80.1', 1);
  t('3.5831808', '1.2', 7);
  t('41.08469075197275390625', '-1.45', 10);
  t('-128', '-2', 7);
  t('-5277043158.170301334052958483', '-12.03', 9);
  t('100.31024025', '10.0155', 2);
  t('262144', '4', 9);
  t('-2.66450474490105494016e-70', '-1.86E-8', 9);
  t('1.58541338622748269695158625134808009096449e+21', '-131.83', 10);
  t('1.69e-14', '1.3E-7', 2);
  t('81', '-3', 4);
  t('1.3979045762098993055105649e+25', '327', 10);
  t('-2.8334269484119140625e-17', '-0.0145', 9);
  t('8455365204.69607', '96.7', 5);
  t('4.826809', '-1.3', 6);
  t('-4.2027e-14', '-4.2027E-14', 1);
  t('-3671885391606530844.02199', '-5163.9', 5);
  t('1.4833', '1.4833', 1);
  t('1078702060.96', '32843.6', 2);
  t('16384', '4', 7);
  t('0.000041046707114327285209693732389121', '-0.185729', 6);
  t('9.04141586678594150656e-76', '1.73404E-19', 4);
  t('7', '7', 1);
  t('-5.408864751631992324037382349788164206309e+39', '-25989', 9);
  t('3.6213725246884329693766314512921496510839580087890625e-58', '0.00000180255', 10);
  t('51.58686976', '-2.68', 4);
  t('32562222.2784028467808485507564561204461862636001', '5.6399', 10);
  t('1.5269627878770126091369423832739776731492122624e-24', '-0.0041532', 10);
  t('2.27128515349184346452713121698128944001e-82', '-6.849E-9', 10);
  t('788.12330352545906108118941549114886272', '2.59298', 7);

  t('49478904727733663575.051582184770446498259493718971258958961046708278746324459568256251259157238318185790764612200843144821137580796334089638532287323488178607687532035618686192601481827902773410791645057647691925965756808571011539462511641266331263543939501901006735272661302011976870015625991111560481188335365459566760212844848008439578745211448299643484693328447464460429576345111356618402802188681216', '-1.60380', 96);
  t('1.17647340482075248419254427344376045314316013640615098803852731448350843468375614975229837000370025634765625e+81', '-1312.5', 26);
  t('7.0251816660378376802360594609084621021532608383100803495211258733288152327693358768741786747381360859889479870288549139760601303377485234176e+87', '-2391.76', 26);
  t('85.717123455378345835968064', '2.0998', 6);
  t('7.8189586726154755713303078223442071754329e+40', '187', 18);
  t('9.145283009860369082115124525217171723626500086794105722560960199131059600008740516579605574000306840952542591838726900925804876261491646795282472377021036430500854987560433039755667787412917103148092128371355207976203112911565783359385741850905996481e+41', '-6.4113', 52);
  t('2.05766289512932241998482631797034621069017517962263802542480365981262536498089637707776e+86', '26', 61);
  t('98190573319064.214163360748608660888346857427032348178060040012159036566957731914573312588336568078808326155328828092023492584783227218123097647282496144318320115860427194595696623235777670393347299594686642872330179664094829207439052010438267983369137644871417464586087393161049', '-1.6293', 66);
  t('1.392598192044227355988772015943279484446048049140235246881310663971517522995991944491655264930301742862780134225258915488062891063359354742689683631308848403923615202977361e+171', '109', 84);
  t('1.5469996446227372120183605762031536360861258626528255060075983309687394111102266369973688377324297491658050323998067994446097153176237864040180207896595918522219595889946775556217e+64', '13.37', 57);
  t('417.53905413413116367045797', '1.3', 23);
  t('1.624290817967129588779052613030846258081e+39', '79677', 8);
  t('6.613998972307900952958925063693292646912001305426127894831260851822286968476028114198011923029274945784114379928252947652708965396453170282038903005249536e-263', '0.00000908', 52);
  t('17179869184', '2', 34);
  t('9.7756260861181096053516970094092017016032995016954767790504110039917368643756356818044285001644705953440430468819149794081183248976623767084182482857941562203266736374606395800621259352114936429301555533971456e+141', '131.6', 67);
  t('4294967296', '2', 32);
  t('131621703842267136', '-6', 22);
  t('7.9228162514264337593543950336e+28', '2', 96);
  t('-1', '-1', 41);
  t('5.0096959237841976223015468271587016074738912992534254294320102857326166341642003475731604090433896328441157235303061782495282575910872792923981113856240569836427242032573810996210128117467500761653330557379347005097932959928243111015057764620250811725789666461853189934772359267843261626881e+197', '140.9', 92);
  t('2.404907604760405225358828131112281116032698930082119547604265954848982041717359046106827774346003151904701415424e+111', '32', 74);
  t('2.69721605590607563262106870407286853611938890184108047911269431464974473521e+22', '-2.7', 52);
  t('18.174', '18.174', 1);
  t('1.36891479058588375991326027382088315966463695625337436471480190078368997177499076593800206155688941388250484440597994042813512732765695774566001e+43', '2.7', 100);
  t('1.25668660115116491467662710973545031109788152235501170924313646703423681882164573441433515503588152722789494580469301248e+72', '34.2', 47);
  t('1.56438420672167084519979701497174364597412396629019101951505612890666817698608365177006994887516584111152450449887451995069033040411213741933147153316594951835865202053237255800944727011696760985308614176976019463960094165747239428884158901702435389388132623055120422260406418255249330452909476806656e+299', '1308', 96);
  t('-1.366174308182025187777528918308146941320736787278460889575749404867695325446359807161410602138908197731224033169833047036778978022709170976735608988099868310813030127497812540494011203545816852288666658682985428600019476831917843e+58', '-4.83', 85);
  t('2.966806088806219456880539423158152815514806645521282752514016241570099725287916962770395912230014801025390625e+28', '-5.15', 40);
  t('8589934592', '2', 33);
  t('3.92012913845865034844198476089476040042936801910400390625e+28', '-10.5', 28);
  t('6.26204859896956162110746451442214931102399028194694661966016701109504299345041406486245804277657094315831017177064948219797627444240188709011456e+72', '10.6', 71);
  t('-7.4010922571797388655581604684527426699437277457499992187268655958229686891933340740793991046504324130585438416249947232728346081088006867719609592595610095069331022367447133421906806310032697675369409918188139600222703781482594772971289070262621142441874895437059567046831661495957273728351051362944953711082346089727330717918651409489163662706555003193741825901442633007312863953564385259087744338361955671878877893732351367361843585968017578125e+364', '-31956.5', 81);
  t('6.192736416792149755132475219493405669399778858602560762871794455317449717448580836229845787580888290473782381059983387805505374173265513677435445890655671110840202944429668728225878544903602239e-270', '0.000319', 77);
  t('2.321827472731847506187210910989207010589651530892302311791322352855446782760907718473019283821056753242967949096791603718944064726033664384082861996009850744961116407128920163494805886357836270766853975184523023642990429569615539676822828070446559161161124635674389704743948383107212919966543969471039720464085404567773913503635201792672811032312063265454094042341369554833714443413857449688164664657173315113983032808001e+220', '159.83', 100);
  t('4.953736650985698001275587761883016173298229503362193331584010106486052779761329981195411366329068322677550960044762649449732919438910119549694010717647780952298320287365138145210431380601324547389442337520392258009e+27', '-2.797', 62);
  t('-3.181403788572786716059998378326698266679069780899509454959934125355133265193e-226', '-0.0000057', 43);
  t('2.2481244795444847556680083178853571141700204725943661497539185751725919215990484669445118877448657555202573745174750533635052209260608084995835891707958259234271442982681066779696130155696868114346639852814575962457281749995256623713554214237458259176913852401064878760784275382146658972801541462348779288776844691139086889351786721e+179', '229.03', 76);
  t('1.09544203269304121665791452848994634135137143229447033238750572549632571476402641497730897157401184093614916314911584167715115647604065370616785146866296916632437569033863034458711783186506727022529378159297811462454675308057682780534190256280910865229376785975491743196806996272790525705427972427376758443312371576466422700240638756574402393677191376501159420836822933903938571935476091044876631814554274289087939783553959776485376e+31', '-2.0436', 100);
  t('-18437563379.178327736384102280592359424', '-2.4', 27);
  t('-9.9938258857146531850367031e-85', '-0.0000000231', 11);
  t('2.45673281757908854552870925433471410070214400334169452520459965621921664416713287959754208329284439619509099451538227140937811535553399003066102795766261859069268099803838998995339331311912275460846260697085073312049264485545427543086425664188963659692922005683716104197951717376e+78', '-6.08', 100);
  t('-1.270210714458885172439348525985048001607770314656078008119108251089872343340175575576722621917724609375e+24', '-4.15', 39);
  t('1209.5', '1209.5', 1);
  t('1220703125', '5.0', 13);
  t('2.2282376678496302739664556233420633e-74', '0.000000006553', 9);
  t('2.218088039378445352751296336694731578146959285984085545221611060271486214348079661041003283535646898758718143100202167951881147633680620698026648392938984778649e+159', '893', 54);
  t('2.10030968607618842407969732186669026418164521383378412734005003194384320968659491239802049483617846390172315534146347011074332799904332281963786542263761412332465321180532568441958976108339460464558490863953503079336356338135760894630619724493454777575270501253639608281424341291617698518141921861556244699796561591915201261004739209197698527393965924844797945393999012105081714167479170123104969e-75', '-0.16053', 94);
  t('2.5812157101662325924336748344070521698410946762506012782923924916938485004644329931151074914373301798478548152505612780265485671672098712055026947247254600014458282820627468534250894082132928621697659949721616819635951012296371246281717642419683893063297904005186874144169940879165882368e+286', '25488', 65);
  t('-15.625', '-2.5', 3);
  t('2.9729685996261878754343622542880008372085710279050331084451543295302017053107644303792930816e+59', '-72.2', 32);
  t('8.15730721', '1.3', 8);
  t('1027.334860999330573407176938442293022889', '-1.47', 18);
  t('1.20109289910050514343148045475506579676136202628631781424931463646123976128634333627993215283736800191296195442718065238702508520055513255346146658609301480565930380776363581062628797012612519067602295957218821773367819208495782017345400726347727499503218384384326417098721424389373157748956399698521853695814977704822352568748499324367082691064737392226159651276162776412093080580234527587890625e+315', '8679.5', 80);
  t('-6.00774319642841528092711102659123449286352750007573133369794363392e-33', '-0.22', 49);
  t('-4.399348626310072691340857398671747903319767685984244866658805429802943497233040561059795990923598510826890908258017493538766494536483606174431755682553269424176946343755577041745496403493430254087811474250251864009392456889456030217048416123130922140883015639602300329038985615284964386359900403921860549312983015992252637174331659470716194350399038561704290046291e+177', '-81.31', 93);
  t('-3.05410827582876268390180635074883196222812916959850956275590047122226930793718597160851022493537771838978253152932907298258251555751167476560688772053768410645587233357374279876239597797393798828125e-247', '-0.000000217485', 37);
  t('5.94029530519133658573841651381893916937282220672559984930445801765696530117833330901747112098660456061962952244422396324466169433806822249922825538552986841011e-19', '0.491', 59);
  t('6.30523820693302485362185385323378612453984181696300485861970499164041182778517073604560519798249286847423997356860449832519236757706844668669858983416671702949907296869514603475482872905487176008793424936405673085021973514041591620544995041842415714193527926259387876594710020672351902248919941484928131103515625e+217', '-207.5', 94);
  t('1.3437482898194084408359108790569717431056936319064040016470306278420649755238589129250716839755182586596297106623509972801915917688676116852207e-198', '0.0047', 85);
  t('-7.2125976851811538835264180669311252876757309817127885777282268797579465235996423577991265862898834402822899090972405912847104748143261578733778979456054998346007864343435284069799588577484912513898046820531311312862161018665360468415091050892814436289818311468501837802070896183140379745282036670643116477155731121958797307187300620339307163520531060733946755523082115115646976e+279', '-767.6', 97);
  t('-3.6014175419419889979825978713404525121917028354184157603813105843373968813713708575013260529677555144144724689493141661197266911743582631754736205113266715503341523389605232359965730457026715491821102337726896541765297511282920487936650122950956509166185123335946923710632464810273024751e+116', '-23.51', 85);
  t('864441744884.215763637320205869663958329635842919446190153261333597518371076208416885489806126587561695103959519298009135858834566699504679568405218950836778578735139166130606578479646072479355927241854902595925015466294859320621226069690609744261681250304', '1.404', 81);
  t('2.1740698194376449374483147468467027620784304276072222988616865620688896e+52', '808.4', 18);
  t('3.8358611506121121577937e+22', '113', 11);
  t('7.708142065054490966774434895299862856570563111111152865942369472254266622057906176e-375', '0.0000000000000002582', 24);
  t('1.697054670829812124918589801369405630774787179949418285317226086899030607907838200923128944336540480137932154797404155642362243064347816887752994688507537824596616056930370261936449615792564115353680658971035397858675806419291627327666883921628766762017e+153', '35.3', 99);
  t('-4.710128697246244834921603689e+27', '-9', 29);
  t('3.2505902718449668133084058626546515621706592677748264541998335302067693284031047914414370910433910917853346703702752017311701370087768406221174636457161634693311184118695477205673579866365261461786447054994132502457715569269165081396243597525386428206972507983546973520886724081775555138991647659604473279e+52', '6.8159', 63);
  t('10125003344964.40019736846338763496828575459544161517257186895064610033021488387585533013795387142569984', '1.4', 89);
  t('-1.4631650542473900277009650505078819321358806229e+27', '-26.90', 19);
  t('0.000005295825289418576524939548312590198218162176', '0.468', 16);
  t('-2.13825209215512828440269455240169347772084407562319954177735966744864590012014618496355742013223900851293500036195697397558882293948621657334016332630442341312358031641684225772703692879077286212410249112292259617578818732032e+224', '-203479518', 27);
  t('5.444517870735015415413993718908291383296e+39', '8', 44);
  t('-33338.549041729', '-33338.549041729', 1);
  t('2.7214030202002235259872217354276868322505756236796101744750922254072943468398158090938361760503196297567571421573431913968089480080443473498108606049915577803053551131894096917430267992139228904833591260925102332702863889641131058350845466180073805176058062062511496305436593986927276878340795001904960142635221849217162104418149855782368706167456898339731322626805338533602069310753044118499679070391659043014863756212342671519561753648641528938961249781667596503442752009505056776106357574462890625e+355', '25403380.775', 48);
  t('-134217728', '-8.00', 9);
  t('6.4919612772480684272077477598407081955951769084986853540654908040236664132468736e+31', '4.6', 48);
  t('-408518.488', '-74.2', 3);
  t('2.6122383559387081296537383480965114437482497294603331038492821930754511334614082451247995976011754175948033208350877260508339210350470971773483611262598975948388081226982520139805939801386768827832059816233575763932242012655210850575522139572300856461682889553829554870249034765972111622144e-53', '0.119534', 57);
  t('1.88522188856624562372708154656751441041381726119427387599765763695834615013232142421520586436232009071932931843172032205056661409003338693915273301223126561630347970098203184279906247655542800284362476149094060315694708604230353242909365032002466875120342181047689475694706227284673985251088619705863125778538982501735383260557349198668384156297528069830916927729430864813333996813440323955895348081117644429414429531206777610934118661932545955356203081687958272101e+350', '1396953.01', 57);
  t('-7.4641079962956705920034912695642465787383566977879750308559196923222402311665962335871104200586964218306687527948988363660881738310704671349709669487610144905689251123467238632699575375039822099609504292687413249132266915915320573775450823119373280252201183083672896166447786672561808861768782076443312543759437244643146889964363933455672502577964296155349186564340392078021387512970362139779496580009767098175116221996336815276032e+365', '-425417.2', 65);
  t('1.915651048048760989675223006710502095760272088541579446290639477213455428121307995153274075329786132157611746651e+72', '363227.051', 13);
  t('-7.65032026817518248876564402449082201253180407179105772287243705588834467106246226864795794350056707139677395342984049031608196167e-158', '-0.0000000000008205956327', 13);
  t('1.5286700631942576193765185769276826401e+37', '7', 44);
  t('1073741824', '8', 10);
  t('-7.492788722116264058568890205770570774859100309722464309788118049412229345772411480224300523702767256086276612012745150464681862935165491623598382139203570285583278591609354972796097254467143774807748614976801272704624819215123157317922738427454525033948883547809409740884056108275497786621497433099431568320476275959806764951584862378486525479494452910448127181400467328943711415916384184330110378180608e+38', '-2.6742', 91);
  t('-9.9615483925456314514941835282258027490657130453558630944504898425429565393128463181577359582035968e+22', '-34.13672', 15);
  t('1.399922200750534682219817715626042346444265577310152832656021878942015248035909712686518235139333930615671437276362060231490721869741275050613538630628001507198150040959125681675660226363983554975062957910826317184614904091530578807538850671607301864207908455029708464592014690257839027200102495492869235566493012545340996194182688131148892605629158333606816661843824439253963260085811270982958376407623291015625e+181', '8668.79595', 46);
  t('1.1972515182562019788602740026717047105681e+40', '9', 42);
  t('2.809433795080632851709145012225734908409054034941321241601987773815883609706743541544402166343924517939889529050536360240983658675834106822309237371584880050176e+67', '29.26', 46);
  t('-1.130764362193076554708917348657069436161383119992010510285626112959278426786936465605907875370782367006311207257151205709430547608789956273155866459033031412425747753017041228269946019068798157376004812291000927158419091251680264057450083682494989856683271885476193403159681085354715517423876371812000883161101039915298940406738346701704184093773206563783365396974283025554248166993067633181013835776e+229', '-495.16', 85);
  t('-3.077228040283419501485435148888847716813106712632276671229710377509021026817676725408321326107948661263691709915698886286976007892051135335651197784824163323643144277812710153968406419716183905139635176333083510237302643593683953953477494067043109636077672667948180821005422658784541128359007025402810248257138220199195378601727145592714535208968044372704483391015842523268884989073259915535384596690721656759730286732238685451797332150695395077171985430070293527840123294365665570804795301448909420756992e+418', '-5398620298.48', 43);
  t('2591500643510.8703986391408896', '35.620', 8);
  t('-3.35390966929685775566428232543003229321430942684482327393890454463616e+26', '-6156.933746', 7);
  t('-4911940958770.721866380729238693889243243167048192', '-25.7122', 9);
  t('2.09670982511886308779364382461226064522913244291274975553909045464953985730555995547122852179186171376596687978705841751285402054060384366505333092240660435368377833186306191605049e+179', '2093', 54);
  t('-1.3120135409024675389197377e-32', '-0.0000000000235855553', 3);
  t('3.06580286249339590556181770635869764259023245764469206004664101e-78', '0.0061', 35);
  t('-4.429293345196932276984549040219058245905488651609667018402382745119170674779904528756935495114384894434070998056621967747005022674509457845620395626349316891933712941619646377506486625368600115937614507993729672492431012578899815005421568e+37', '-32.051870380', 25);
  t('1.299270565684257922619246441017034698039805547412943905460754496572097300952165266370194652051376805296279609746312644347199238836765289306640625e+56', '-18.85', 44);
  t('1.566108714385843340192620380546546555641735293434509574033302065853656477829384746549085411912657151882140111651595251043453318950397645765893613409097202898027341727466585518405768283017471302953847629952837298920882176e+24', '41.0192888808596', 15);
  t('3.23590932372253866812859949454563891250217856922745576253641197253435665205998361592799809e+89', '2474977', 14);
  t('-9.7368504802272205153595678239454304952368475046930119079817985916928e+67', '-12', 63);
  t('3.44552147465294110719732986332367243247925798357929806000836849e+62', '-7', 74);
  t('128', '2', 7);
  t('5.65251025822228531444194278761233641822281792819659230058333528631972497846690920646507426317458140864925211864848848044857090776165664263450974899867329254381660849739991179791628529582217569168185463633540183106805961127540063207424e+251', '-96880334819820', 18);
  t('1.214529036103353599078303402353988354368901290518483617131872964733929620706476271152496337890625e+43', '6.5', 53);
  t('-1.3089738102615705488950036718911116878301386483323342409513651601949943841095689259436086715966084434967103664037924413474102289403813212913777522181382481044473580351805322191321973e+181', '-78533', 37);
  t('1.252453015827223091648143056289e+30', '47', 18);
  t('4.593884321465082753417847006029094939767153045428871371185146539475234934788078659649525017869031479813190380591953477672897146298354193774199686172307595609685062957351072442717423824419651156326307969688849329e+172', '34973.7', 38);
  t('6.4993779802563991313942147088910855038980304493628711994552936552675080091715543868582664478495863312478503560216160496462563144213423384672637506819444007554298505039521124189159003567749810227911510054190116940785207230365438230686589958633589792785739898681640625e+25', '19.527362557915', 20);
  t('22876792454961', '3', 28);
  t('429643228077338878294.90980773552148447232', '13385.0992', 5);
  t('-7316.89', '-7316.89', 1);
  t('5234016485162380478.15409743331356736233224871362317808951664029477472141754368438049197212877764328120319129016485361', '2.41', 49);
  t('-9.18448171920669942200338050009727274073573579465100492416464784644248864677562331387086995022599687737054074140424187879986024804042070022603490146718362719056788563440317890910682257768632925460866552260450415793672740716743856864948011399578402033107962721324762764871499139470157293091799845778141752390211181570286270362089589942764421437922022663206638528993662330911966459538905107421403529415706249287725923857606476849002798755812680628308770502507877888997449832630062999974358780733e+247', '-11613.6333', 61);
  t('2.648409294963988844015616e-24', '0.0000000138355056', 3);
  t('2.40863472207038829716951461596425356200891165327233287498341065017430044564718221477455813374032856667895504085743031089347856425830213625711815509685491800628989456526993157007511653813393067316060298253585492583958097263354394562221211900373157618798878275619835293896301261289133749847496154802330721946407650402052956106639135229248180394944200385403412146150497882700273381421471626181859523581299015700836664867216504968658889499860010034200157321932124862289179191239642591047175122517733014555125735104472403712501002006088016431555682009e+357', '6337.73', 94);
  t('-3502571449.82200575261531309080576', '-2.6', 23);
  t('1', '1', 25);
  t('-1.1809125003067061836785344667120764389443931749265317457525459548456128442105786285468877390219672806038351135283671645849621670658638427296075188535296e+27', '-7.46960', 31);
  t('-7804725.584345565904628551916716032', '-1.8', 27);
  t('652.68343537143736977925527144667849979675453569710942641839519790994481', '-1.1', 68);
  t('1', '-3', 0);
  t('4.26825223812027400796974891518773732342988745354489429495479078935112929549619739019072139340757097296812815466676129830954465240517595242384015591919845376e+55', '3.6', 100);
  t('1.143346541001259437465016483752642875340200408207372315398374410086504054045958634479945641547514904516533919702060746490155830653988883705469429546657675670673347514271976875109918113055399954802405065425692486675365016543448426931378922954782957026636988416e+190', '-388997.14', 34);
  t('1.391479860226959780852496779461260634580951060711489905798047445581119629509685004732118558920853318033429289719854192097911392274604744812177542476869031306237630906988638098552100901480110526866015997536096860367785128698535455722265589251495705036009334110148089566472594008624482378925581538366245059147829461639938451131334656e+162', '785.994', 56);
  t('-2.039429275458086101347617297850948854294429450037773018607110681185721614063201210356599410953e+93', '-27753', 21);
  t('1.8072708076715752612606017458825273480777271963492548142293548724970849500539908571180569517655791008990836891664536992680448e+43', '64022.984420938', 9);
  t('-155568095557812224', '-14', 15);
  t('2.18753957001465798602093911444285519704647324430365134831536108015616e+44', '4954.46', 12);

  Decimal.toExpNeg = Decimal.toExpPos = 0;

  t('2e+0', 2, '1.0');
  t('1.6e+1', 2, '4.00000000');
  t('6.25e-2', 2, -4);
  t('5.0600621890668482322956892808849303e+20', '907.27', 7);
  t('-7e+0', '-7', 1);
  t('-9.01e+2', '-901', 1);
  t('1.016984074247269470395836690098169093010136836967e+39', '21584.7', 9);
  t('-8.983272e+1', '-89.83272', 1);
  t('5.308416e+6', '-48', 4);
  t('3.83432050166120236679168e+23', '52088', 5);
  t('-2.679971527468745095582058350756311201706813294321409e+51', '-517889', 9);
  t('5.067853299870089529116832768e+2', '3.47508', 5);
  t('3.48822062687911109850066182676769e+32', '4129', 9);
  t('1e+0', '-429.32321', 0);
  t('-4.2773e+0', '-4.2773', 1);
  t('-5.8169306081172252508071119604378757744768e+12', '-66.6082', 7);
  t('1e+0', '-7.0654', 0);
  t('-1.51655708279450944384385164853883404204414169862685507e+46', '-3956084.3', 7);
  t('8.1e+1', '-3', 4);
  t('1.296e+3', '-6', 4);
  t('2.9e+0', '2.9', 1);
  t('1.764e+3', '-42', 2);
  t('-9.3418332730097368870513138581415704704611459349313e+49', '-356673', 9);
  t('1.517108809906561e+15', '79', 8);
  t('3.1063e+4', '31063', 1);
  t('1e+0', '-21914.49416', 0);
  t('1.4586250332983909737249e+10', '49.43', 6);
  t('-4.208092749838142546109102616048103440952842087045576022556672e+46', '-4577028.48', 7);
  t('3e+0', '3', 1);
  t('1.4301918832998497740081358663795354088849575358074712812593899025686055664154241e+71', '784195396.7', 8);
  t('1.914994057877243921847839041839022064004621019747303000421532785660701999616e+57', '2315715.86', 9);
  t('1.40379741624014849127481344e+26', '804', 9);
  t('2.1305364420464979969795870610432e+21', '18437.32', 5);
  t('1e+0', '-48', 0);
  t('7.018e+3', '7018', 1);
  t('-5.8149737003040059690390169e+25', '-729', 9);
  t('2.537640625e+9', '50375', 2);
  t('1e+0', '-287', 0);
  t('-1.879616311308566413901e+11', '-179.81', 5);
  t('-3.73314280039567349e+17', '-3269', 5);
  t('8.503056e+6', '-54', 4);
  t('1.114538684361769559136325232186960408417483078419495901986816e+28', '3205.4344', 8);
  t('-4.27504205e+3', '-4275.04205', 1);
  t('-4.18195493e+8', '-53', 5);
  t('-1.32651e+5', '-51', 3);
  t('-4.309348e+6', '-4309348', 1);
  t('5.2697770766776504576e+19', '-7259323024', 2);
  t('7.334473e+6', '7334473', 1);

  t('4.2587208337796928798976e+22', '-674', 8);
  t('2.5398714208994744315474847375553672678449979949471296663472852498581058264323e+76', '82219173067', 7);
  t('3.8595867893817789e+16', '38595867893817789', 1);
  t('1e+0', '6607929.612', 0);
  t('2.03571124747179512510686558355453870501244709050625e+34', '-377727882.4895', 4);
  t('2.45293306092383728169089251608498030232760591201e+47', '-838901', 8);
  t('1e+0', '-3', 0);
  t('3.834566459566834602823771544700625e+33', '248845085', 4);
  t('9.9603189655699799982630929090395077799549456e+43', '-99900649454', 4);
  t('4.96981290961e+11', '-89', 6);
  t('1.58714182711801961914035004098749014540213215369889902124332186866896819339082286680313155541911581006508901525889227579394470766224134369e+74', '175595288.8066529', 9);
  t('1.326409999413464946341631126231535779147856072448189797080406801e+19', '-60348.91227522707', 4);
  t('-2.050107090202653994550415295865311714194868710425929507925126964581376e+69', '-72837546696276', 5);
  t('2.0550048143314549586011369057890711537772093948321e+49', '-2129135057417', 4);
  t('1e+0', '3943801.89538088', 0);
  t('2.17438998607457e+14', '737', 5);
  t('-2.273581752472e+10', '-22735817524.72', 1);
  t('1.51891070159203e+12', '1518910701592.03', 1);
  t('-6.42199562432576e+14', '-86276', 3);
  t('1.33010089e+8', '11533', 2);
  t('5.1662108991396663099278667856407361e+26', '-4767525.59', 4);
  t('-2.197e+3', '-13', 3);
  t('5.7261215932713209368576e+22', '489176', 4);
  t('2.63105908732834720740862027188940646962896811537209872829318157804817881e+71', '800488505731', 6);
  t('-7.8151742291e+2', '-781.51742291', 1);
  t('-2.5619022164869008875e+19', '-2947955', 3);
  t('1.888497394256858387595532951034741052100428552247961627495355202904723558656e+3', '-6.592180609971056972', 4);
  t('1.227102111503512992112190463e+27', '1023', 9);
  t('4.762033342402709161761620616017059035607e+39', '86210167', 5);
  t('1.0526899962331857933889886363874543386374239042307e+34', '6374705.107', 5);
  t('8.3043861038534085480857730707894983974895341475449922046323943309856240866914720437665233e+61','7584431.953', 9);
  t('5.4787291089024129877918140548999194405505068584642366052046899220357579072497678119620025119214429570421679076944647727607529032984620150444553526120397e+151', '72419131838243117', 9);
  t('6.1203577224967e+13', '61203577224967', 1);
  t('-2.6742593337474927167986914400257603e+34', '-82827', 7);
  t('3.490938536648870448335810684618787855050175354916141007545163425555957335829e+75', '247548469', 9);
  t('6.634204312890625e+15', '-95', 8);
  t('-9.171271e+1', '-91.71271', 1);
  t('5.65625765123605544067809018296482219675982471958093705254464e+59', '-9093998202', 6);
  t('3.518743761e+9', '-39', 6);
  t('1e+0', '-5420353171006060062', 0);
  t('5.93548365959371477646876280183482286894691258676416e+26', '840398698.65610156', 3);
  t('3.05021119744369888239209417566581849921624834657246971302432870103990316674041818082170961e+89', '-820455414011161', 6);
  t('3.2696530375744e+11', '571808.8', 2);
  t('7.5976962512577721e+4', '-275.639189', 2);
  t('1.902906615566371112511312310657143013376e+4', '7.1760176', 5);
  t('1.518464318469541965652429354308518011195184133938752135708801e+60', '33317743', 8);
  t('2.13200587786233626937431579432326390786198510967574168821917803373975072742648456640669209424089001389646413824e+92', '2444169945946508.968', 6);
  t('2.22147484530834915265639755288996e+8', '-14904.612860817114', 2);
  t('1.492078384275163354935132877205588401e+36', '-1105217899', 4);
  t('-8.7063159741318127407528723363285743789464660953004359884056134456089069164858324456719131936164339926111276712486987242162931090900645752178720048529774274761e+31', '-3538.91793390607241', 9);
  t('-2.13718835678657210508559759338095142848187133522908355458407047497607984468725768078407393313580673815584615486794665109272908382172581587476837516398285148210296241056810064518192239263716914808770258390627580625498767684881468120398755080836616073670778327363332334149416019505078840253645081995145611223104498789608205428495349224542916922658293232780298744501314853052243306816366148311919597221018850456061527891844821902500302159789434488225792e+125', '-43837.525307806238', 27);
  t('6.6067183217034216835242910438119195789788339709493349261334009277024675568767116522521737472677079426635748603259327817717050761351617610783486587612028204471198915744801916514515768618923473461233676712402753179616115705829361991013069927369820940249862144e+28', '-63581.52864975395137421365938883267810362492', 6);
  t('3.4979655423042386318207008930737637967359866053138998526505479068249644854876701482811561341347105355107695596776164641704434681969517211923069874207364828070027467521475982678466251216378786468053405366050900916905682920934733855175647035328921497719019616796867272653837232009633511918699584196136643032274244642212313884180837128156688381283842300678798685499010807063746785776492193558002382725297304258913361e+62', '860658355.93419774133511168973596889241553275604551215135281', 7);
  t('2.43e+2', '3', 5);
  t('1.1341259418236998571758990926811782086077524757263871423863079766086030645673726799115988350322336274469528121736526520039192242333771040454545272536238688950013853620343757910936524944674880346697808001951639020756685105416346426508389541123017097529060283829118103883424462596062875341251599735321937114348037712815556934333412355272212045907182873831909566029976608100685207198148793211027456e+142', '-77969657.57799062937836', 18);
  t('-3.0028686351108853762578194984762919284354267811356443985818417209466959135294437327385632801156377236905276696018118450080884435331856383482308738976558186955588239868161702337334759747439195397392938819582577110829644281561085620841415307488010481547467558616309027516654381610456065415065653113913265920029870297942479155862011879884534002975129045737955826661617772520189976935251896803639204375428428030749e+58', '-3144155.647351179442684823966727126493552586749', 9);
  t('1.201753576366394485531985516556867241747342491824062593983537252565487129409802473156809713616241453319630220923725569628522288147931699557284946119375022985149050727430775477724150730989713693887368264655960182781852240604839836052804518341222619522792226084796948765314193322314769946060633722353531243331728258530762514582945950418550784e+31', '27540.61485935537676544614263765574563131573245204', 7);
  t('3.37155040678778244943012545203918986928745669295171850695394544390939711301031039954614562269468815944072319615611367890859596537480727045421148118535227427436434942469760108682572253320836320356162649649114580786938367825230347818905248996593031355662611333376e+4', '-3.68110925219710489918033269313682', 8);
  t('-1.157385562678413511184308898769846329872668259271078544423253372613586833033423503791374542602879075132311305638124240471079826030985837104460703581522437292662638238646090030670166015625e+186', '-7784665', 27);
  t('5.7216584769709576633639728428105243709762039786762015993329350164736e+39', '-8697220540.1137988', 4);
  t('3.605080945893673836738483946060969335788919921875e+3', '2.48435', 9);
  t('-1.36240450766284865053749603054987406024313651234922549391481734582321604854817272341996695098685037495173906240642212299142117181990034526636150309000822666516664160854969363947955605850108340373472205128364217170231566407701356867606602700453971968987168686696650431666690097496345998516797552915475256643599049488322243247996920458277719083301674288561258518312772229949771006312491204843363917607453589630126953125e+161', '-3009460118.872263767875045', 17);
  t('-3.8433712973156659829312033596061261378003902713922379084688365643813353176050190118421109324160544394293788636677094907930777317141076841010276342345029919613746057639577024242309077555840877645790649554834063514938596292845365250946008190566774814236719875110926196028928250892121252859736673222597350006754961294389605594296221696e+157', '-271616.155606', 29);
  t('8.880125173267286184293229956929e+30', '-143903', 6);
  t('1.2933690030633029978317864458950890700698349292432976269804895466563737201880939364145851524416113479708518067001609042030295271670130750045121e+14', '-7.62048571', 16);
  t('4.49118544044091499255717578905704133184433268994940305778505055836352880272044837700029253390614132345790751503571600332872e+26', '765808758.66956931514481639498062905791538', 3);
  t('-1.128193052103617375313057818469912250487118062091774976e+29', '-646363.29196', 5);
  t('-3.852842458277771128223673975490918109e+4', '-38528.42458277771128223673975490918109', 1);
  t('4.47317529320544012507678955001652121720459770030481180787405405093720278140090456738247550862238112656659162732461713512251836889602737214995151906586537236627456e+105', '5932.48', 28);
  t('1e+0', '-13.97065588562499', 0);
  t('3.600545803275794460618862354918801e+11', '600045.48188248151', 2);
  t('2.569380011313067359691205257942517240115628560689496193366261158908199626359277748729466511294505469537479575614097255886064905027804073982599354300474897943e+129', '62080.7', 27);
  t('5.818461494039866039726986100228248575472782020261098417686712139322808092966875943575012778303872663798273362332843148103605110550575434794509775866235877112057573531799569068637521409971342270836350080123911710082519119058044202449706075905917444188007277340785315892319100567144025511009943278859996309146349108129588009524947393206783018441414022970002633906397294399684375635228021017590940658971148436650082905540838555648e+102', '6398.880593717242', 27);
  t('1.34971356455758975292534311187056647652453780619004003032217331555704261585125261846040712896856946037144900053640398406244356726653396169868958721875222540716653887768486266703330919098362921e+11', '4.1528548909', 18);
  t('-1.77396388128500114222945974047586725531284545911409446577e+56', '-913', 19);
  t('-1.23493440106890737225414566980096565722401440419412087747352443276574932419181598097379536554237254841727093836607763912665203669933311010821309594074994067411700657618950544400833319706872315553946358432623522564358319794505880365894233180227408889879677018966524371664896e+230', '-90520083747.46', 21);
  t('-2.17525001942067723150294558531604730954322658381064353285681098867925681081506083705983156066163186746091443633027643554689317160117967955473175500964155968361524273945120598810972722792061e+14', '-3.121741', 29);
  t('1e+0', '-562610.16179592649796590823394093366548', 0);
  t('5.5399516545866639955830015026176e+11', '14.94', 10);
  t('1.494189395849269188211255039709933309086424259778445906419464942576097148044216376789735318980392750336285644804638743600807550074206128272345650029255016954321611264002141919840462369550905098763723254901675135787504979910497931539962354019230845564318816091666473025536e+126', '-32698949771.110178432792', 12);
  t('1.1504940871276742926708823617505372960241390892442547940194260102743306128298973371802547471453755938653944600792141533514422490556963535378521856840746722206160260148176604222079226186281680715577736316488196108040509176925372372483300649927835887344415603493563915019264675154039059770309142781930141352329958156926976e+15', '-8.7358521345995835476', 16);

  tx(function () {new Decimal('12.345').pow(void 0)}, ".pow(void 0)");
  tx(function () {new Decimal('12.345').pow(null)}, ".pow(null)");
  tx(function () {new Decimal('12.345').pow(true)}, ".pow(true)");
  tx(function () {new Decimal('12.345').pow(false)}, ".pow(false)");
  tx(function () {new Decimal('12.345').pow([])}, ".pow([])");
  tx(function () {new Decimal('12.345').pow({})}, ".pow({})");
  tx(function () {new Decimal('12.345').pow('')}, ".pow('')");
  tx(function () {new Decimal('12.345').pow(' ')}, ".pow(' ')");
  tx(function () {new Decimal('12.345').pow('4e')}, ".pow('4e')");
  tx(function () {new Decimal('12.345').pow('hello')}, ".pow('hello')");
  tx(function () {new Decimal('12.345').pow('\t')}, ".pow('\t')");
  tx(function () {new Decimal('12.345').pow(new Date)}, ".pow(new Date)");
  tx(function () {new Decimal('12.345').pow(new RegExp)}, ".pow(new RegExp)");
  tx(function () {new Decimal('12.345').pow(function (){})}, ".pow(function (){})");
});

/*
  Notes:

  n to the power of 1 is n
  Anything to the power of 0 is 1
  Math.pow(2, null) = 1            // NaN for Decimal
  Math.pow(2, undefined) = NaN
  Math.pow(2, NaN) = NaN
  Math.pow(2, 'rferf') = NaN
  Math.pow(2, []) = 1             // NaN for Decimal
  Math.pow(2, {}) = NaN           // NaN for Decimal
  Math.pow(2, 1e4) = Infinity
  Math.pow(2, -1e4) = 0
  Math.pow(2, 1e-16) = 1
  Math.pow(2, -1e-17) = 1
  A negative number to a non-integer power is NaN

  http://es5.github.com/#x15.8.2.13
  15.8.2.13 pow (x, y)
  Returns an implementation-dependent approximation to the result of raising x to the power y.

  If y is NaN, the result is NaN.
  If y is +0, the result is 1, even if x is NaN.
  If y is -0, the result is 1, even if x is NaN.
  If x is NaN and y is nonzero, the result is NaN.
  If abs(x)>1 and y is +Infinity, the result is +Infinity.
  If abs(x)>1 and y is -Infinity, the result is +0.
  If abs(x)==1 and y is +Infinity, the result is NaN.
  If abs(x)==1 and y is -Infinity, the result is NaN.
  If abs(x)<1 and y is +Infinity, the result is +0.
  If abs(x)<1 and y is -Infinity, the result is +Infinity.
  If x is +Infinity and y>0, the result is +Infinity.
  If x is +Infinity and y<0, the result is +0.
  If x is -Infinity and y>0 and y is an odd integer, the result is -Infinity.
  If x is -Infinity and y>0 and y is not an odd integer, the result is +Infinity.
  If x is -Infinity and y<0 and y is an odd integer, the result is -0.
  If x is -Infinity and y<0 and y is not an odd integer, the result is +0.
  If x is +0 and y>0, the result is +0.
  If x is +0 and y<0, the result is +Infinity.
  If x is -0 and y>0 and y is an odd integer, the result is -0.
  If x is -0 and y>0 and y is not an odd integer, the result is +0.
  If x is -0 and y<0 and y is an odd integer, the result is -Infinity.
  If x is -0 and y<0 and y is not an odd integer, the result is +Infinity.
  If x<0 and x is finite and y is finite and y is not an integer, the result is NaN.

*/
