// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

use chrono::Duration;
use pgp::types::{EcdhPublicParams, PublicParams};

/// Map expiration Duration to OpenPGP style seconds.
///
/// If the duration is too large, this returns u32::MAX
/// (such overflows can not happen for durations that come directly from an OpenPGP packet).
pub(crate) fn duration_to_seconds(d: Option<&Duration>) -> Option<u32> {
    d.map(|d| u32::try_from(d.num_seconds()).unwrap_or(u32::MAX))
}

pub(crate) fn algo_name(pp: &PublicParams) -> String {
    match pp {
        PublicParams::RSA { n, .. } => format!("RSA {}", n.len() * 8),
        PublicParams::ECDH(EcdhPublicParams::Known { curve, .. })
        | PublicParams::EdDSALegacy { curve, .. } => curve.name().to_string(),
        PublicParams::ECDSA(_) => "ECDSA".to_string(),
        _ => "TODO".to_string(),
    }
}
